//
//  SCSIInquiryCommand.h
//  DriveUnlock
//
//  Created by Bob on 3/11/09.
//  Copyright © 2014 Western Digital Technologies, Inc. All rights reserved.
//

#include <sys/param.h>

#import <IOKit/IOCFPlugIn.h>
#import <IOKit/scsi/SCSITask.h>
#import <IOKit/scsi/SCSICommandOperationCodes.h>
#import <IOKit/scsi/SCSICmds_MODE_Definitions.h>
#import <IOKit/storage/ata/ATASMARTLib.h>

#import "WDDevice.h"
#import "WDDMcommon.h"
#import "WDDevice+SendSCSICommand.h"
#import "WDDevice+Encryption.h"

#define defaultCipherID			AES_256_ECB	//	Default to AES_256_ECB.
#define defaultKeyLength		256

#define MAX_SCSI_INQUIRY_DATA_SIZE       256

//typedef union {
//	OSStatus		ret;
//	struct {
//		UInt8		status;
//		UInt8		sensekey;
//		UInt8		ASC;
//		UInt8		ASCQ;
//	} scsi;
//} SCSIStatus;

enum {
	// WDC Vendor specific SCSI command codes
	
	kENCRIPTION_STATUS_COMMAND		= 0xC0,
	kCHANGE_ENCRYPTION_PASSPHRASE	= 0xC1,
	KRESET_DATA_ENCRIPTION_KEY		= 0xC1,
	kUNLOCK_ENCRYPTION_COMMAND		= 0xC1,
	kFORMAT_DISK					= 0xC4,
	kREAD_HANDY_CAPACITY			= 0xD5,
	kREAD_HANDY_STORE				= 0xD8,
	kWRITE_HANDY_STORE				= 0xDA,
	kREAD_VITURAL_CD_CAPACITY		= 0xC5,
	kREAD_VIRTUAL_CD				= 0xC8,
	kWRITE_VIRTUAL_CD				= 0xCA,
    kREPORT_LUNS                    = 0xA0,
    kREPORT_RAID_STATUS             = 0xA3,
    kREPORT_SLOT_DETAILS            = 0xA3,
    kSET_RAID_CONFIGURATION         = 0xA4,
    kUNMAP_SMR_DRIVE                = 0x42
};

enum {
	kENCRIPTION_STATUS_COMMAND_SIGNATURE	= 0x45
};

enum {
	kENCRIPTION_STATUS_COMMAND_LENGTH	= 10,
	kCHANGE_ENCRYPTION_PASSPHRASE_LENGTH = 10,
	KRESET_DATA_ENCRIPTION_KEY_LENGTH	= 10,
	kUNLOCK_ENCRYPTION_COMMAND_LENGTH	= 10,
	kREAD_HANDY_CAPACITY_LENGTH			= 10,		
	kREAD_HANDY_STORE_LENGTH			= 10,
	kWRITE_HANDY_STORE_LENGTH			= 10,
	kLOG_SENSE_COMMAND_LENGTH			= 10,
	kLOG_SELECT_COMMAND_LENGTH			= 10,
    kREPORT_RAID_STATUS_LENGTH          = 12,
    kREPORT_SLOT_DETAILS_LENGTH         = 12,
    kSET_RAID_CONFIGURATION_LENGTH      = 12
};

enum {
	kUNLOCK_ENCRYPTION_COMMAND_ACTION	= 0xE1,
	kCHANGE_ENCRYPTION_PASSPHRASE_ACTION = 0xE2,
	KRESET_DATA_ENCRIPTION_KEY_ACTION	= 0xE3
};

enum {
	kMODE_SENSE_DBD					= 0x08,
	kMODE_SENSE_LLBAA				= 0x10
};

enum {
	kMODE_SENSE_CURRENT_VALUES		= 0x00,				// 0x00
	kMODE_SENSE_CHANGEABLE_VALUES	= 0x40,				// 0x01
	kMODE_SENSE_DEFAULT_VALUES		= 0x80,				// 0x02	
	kMODE_SENSE_SAVED_VALUES		= 0xC0				// 0x03
};

enum {
	kPOWERCONDITIONPAGE				= 0x1A,
	kMODEDEVICECONFIGURATIONPAGE	= 0x20,
	kMODEOPERATIONSPAGE				= 0x21,
	kMODE_SENSE_ALL_PAGES           = 0x3F
};

/*!
 @struct SCSICmd_INQUIRY_StandardData
 @discussion
 This structure defines the format of the required standard data that is
 returned for the INQUIRY command.  This is the data that is required to
 be returned from all devices.
 */
enum
{
kINQUIRY_PRODUCT_PRODUCT_ID_Length = 4
};

typedef struct __attribute__((packed))SCSICmd_INQUIRY_WDData
{
    UInt8		PERIPHERAL_DEVICE_TYPE;				// 7-5 = Qualifier. 4-0 = Device type.
    UInt8		RMB;								// 7 = removable
    UInt8		VERSION;							// 7/6 = ISO/IEC, 5-3 = ECMA, 2-0 = ANSI.
    UInt8		RESPONSE_DATA_FORMAT;				// 7 = AERC, 6 = Obsolete, 5 = NormACA, 4 = HiSup 3-0 = Response data format. (SPC-3 obsoletes AERC)
    // If ANSI Version = 0, this is ATAPI and bits 7-4 = ATAPI version.
    UInt8		ADDITIONAL_LENGTH;					// Number of additional bytes available in inquiry data
    UInt8		SCCSReserved;						// SCC-2 device flag and reserved fields (SPC-3 adds PROTECT 3PC TPGS, and ACC)
    UInt8		flags1;								// First byte of support flags (See SPC-3 section 6.4.2)
    UInt8		flags2;								// Second byte of support flags (Byte 7) (See SPC-3 section 6.4.2)
    char		VENDOR_IDENTIFICATION[kINQUIRY_VENDOR_IDENTIFICATION_Length];
    char		PRODUCT_IDENTIFICATION[kINQUIRY_PRODUCT_IDENTIFICATION_Length];
    char		PRODUCT_REVISION_LEVEL[kINQUIRY_PRODUCT_REVISION_LEVEL_Length];
    char        USB_PRODUCT_ID[kINQUIRY_PRODUCT_PRODUCT_ID_Length];
    char        VENDOR_SERIAL_NUMBER[16];           // Obsolete use Serial number page.
    UInt8       reserved1:4;
    uint8       reserved2:2;
    UInt8       reserved3:1;
    UInt8       reserved4:1;
    UInt8       versionDescriptor1[2];
    UInt8       versionDescriptor2[2];
    UInt8       versionDescriptor3[2];
    UInt8       versionDescriptor4[2];
    UInt8       versionDescriptor5[2];
    UInt8       versionDescriptor6[2];
    UInt8       versionDescriptor7[2];
    UInt8       versionDescriptor8[2];
    UInt8       reserved5[22];
} SCSICmd_INQUIRY_WDData;
typedef SCSICmd_INQUIRY_WDData * SCSICmd_INQUIRY_WDDataPtr;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0x00
    UInt16          PageLength;
    UInt8           VPDPageCodes;
} SUPPORTED_VPD_PAGES_PAGE;

typedef struct __attribute__((packed)){
#ifdef __LITTLE_ENDIAN__
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
#else
    UInt8			PeripheralQualifier:3;
    UInt8			PeripheralDeviceType:5;
#endif

    UInt8			PageCode;               // 0x80
    UInt8			Reserved;
    UInt8			PageLength;
    UInt8			ProductSerialNumber[32];
} SERIAL_NUMBER_PAGE;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xB3
    UInt16          PageLength;
    UInt8           DesignationDescriptor;
} DEVICE_IDENTIFICATION_VPD_PAGE;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xB0
    UInt16          PageLength;
    UInt8           WSNZ:1;
    UInt8           reserved1:7;
    UInt8           MaximumCompareAndWriteLength;
    UInt16          OptimalTransferLengthGranularity;
    UInt32          MaximumTransferLength;
    UInt32          OptimalTransferLength;
    UInt32          MaximumPrefetchXDREADXDWRIETransferLength;
    UInt32          MaximumUnmapLBACount;
    UInt32          MaximumUnmapBlockDescriptorCount;
    UInt32          OptimalUnmapGranularity;
    UInt32          UnmapGranularityAlignment:31;
    UInt32          UGAValid:1;
    UInt32          MaximunWriteSameLength;
    UInt            reserved2;
} BLOCK_LIMITS_VPD_PAGE;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xB1
    UInt16          PageLength;
    UInt16          MediumRatationRate;
    UInt8           reserved1;
    UInt8           NominalFormFactor:4;
    UInt8           reserved2:4;
    UInt8           reserved3[56];
} BLOCK_DEVICE_CHARACTERISTICS_VPD_PAGE;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xB2
    UInt16          PageLength;
    UInt8           ThresholdExponent;
    UInt8           DP:1;                   // DESCRIPTOR PRESENT
    UInt8           ANC_SUP:1;              // ANCHOR SUPPORTED
    UInt8           LBPRZ:1;                // LOGICAL BLOCK PROVISIONING READ ZEROS
    UInt8           reserved1:2;
    UInt8           LBPWS10:1;              // LOGICAL BLOCK PROVISIONING WRITE SAME(10) COMMAND
    UInt8           LBPWS:1;                // LOGICAL BLOCK PROVISIONING WRITE SAME(16) COMMAND
    UInt8           LBPU:1;                 // LOGICAL BLOCK PROVISIONING UNMAP COMMAND
    UInt8           ProvisioningType:3;
    UInt8           reserved2:5;
    UInt8           reserved:8;
} LOGICAL_BLOCK_PROVISIONING_PAGE;

// Page 0xC0 is obsolete.

typedef struct __attribute__((packed)){
    UInt8           Active:1;
    UInt8           reserved1:7;
    char            PortType[7];
}  PORT_DESCRIPTOR;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xC1
    UInt16          PageLength;
    PORT_DESCRIPTOR PortDescriptors[];
} ACTIVE_INTERFACES_VPD_PAGE;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xC2
    UInt16          PageLength;
    UInt8           Signature;              // 0x30
    UInt8           reserved1;
    UInt8           MaxDisks;
    UInt8           DisksInstalled;
    UInt64          TotalNumberOfBlocks;
    UInt32          BlockLength;
    UInt32          PreferredBlockLength;
} RAW_CAPACITY_VPD_PAGE;

typedef struct __attribute__((packed)){

    // This is wrong...Fix it later since we are not using it currently.

    UInt64          Action:4;                   // IEEE Company ID bits 23:20
    UInt64          NAA:4;
    UInt64          IEEECompanyID:20;
    UInt64          VendorSpecificIdentifier:36;
} WORLD_WIDE_NAME;

typedef struct __attribute__((packed)){
    UInt8           GenuineIdentfiersSignature;     // 0x42
    UInt8           reserved1;
    UInt8           reserved2;
    UInt8           reserved3;
    WORLD_WIDE_NAME GenuineTargetPortWWN;
    WORLD_WIDE_NAME GenuineTargetPortWWNForUSB;
    WORLD_WIDE_NAME GenuineDiskWWN;
    WORLD_WIDE_NAME GenuineVirtualCDWWN;
    WORLD_WIDE_NAME GenuineSESWWN;
} FACTORY_CONFIGURE_GENUINE_IDENTIFIERS_PARAMETERS_LIST;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xC3
    UInt16          PageLength;
    UInt8           Signature;              // 0x30
    UInt8           Frozen:1;
    UInt8           reserved1:7;
    UInt8           reserved2;
    UInt8           SlowEnumerationOnce:1;
    UInt8           reserved3:7;
    UInt16          NewGenuineVendorID;
    UInt16          NewGenuineProductID;
    FACTORY_CONFIGURE_GENUINE_IDENTIFIERS_PARAMETERS_LIST
    FactoryConfigGenuineIndentifiers[];
} FACTORY_CONFIGURE_GENUINE_IDENTIFIERS_VPD_PAGE;

typedef struct __attribute__((packed)){
    UInt8           RebrandedIdentifiersSignature;      // 0x50
    UInt8           vvalid:1;
    UInt8           reserved1:7;
    UInt8           reserved2;
    UInt8           reserved3;
    UInt16          RebrandedVendorID;
    UInt16          RebrandedProductID;
    char            RebrandedT10VendorIdentification[8];
    char            RebrandedProductIdentification[16];
    UInt32          RebrandedLANGID;
    UInt8          RebrandedManufacturerString[42];
    UInt8           RebrandedProductString[42];
} FACTORY_CONFIGURE_REBRANDED_IDENTIFIERS_PARAMETER_LIST;

typedef struct __attribute__((packed)){
    UInt8			PeripheralDeviceType:5;
    UInt8			PeripheralQualifier:3;
    UInt8           PageCode;               // 0xC4
    UInt16          PageLength;
    UInt8           Signature;              // 0x30
    FACTORY_CONFIGURE_REBRANDED_IDENTIFIERS_PARAMETER_LIST
                    FactoryConfigireRebrandedIdentifiersParameterList[];
} FACTORY_CONFIGURE_REBRANDED_IDENTIFIERS_VPD_PAGE;

//////
//
//      End Inquiry Vital Product Data Page Codes
//
//////

typedef struct __attribute__((packed)){
	UInt8			LastCDAreaBlockAddress[4];
	UInt8			BlockLength[4];
	UInt8			reserved[2];
	UInt8			MaximumTransferLength[2];
} READ_VITURAL_CD_CAPACITY_DATA;

typedef struct __attribute__((packed)) {
	UInt8			modeParamHeader[4];
#ifdef __LITTLE_ENDIAN__
	UInt8			pageCode:6;
	UInt8			reserved1:1;
	UInt8			ParamSavable:1;
	UInt8			pageLength:8;
	UInt8			signature:8;
	UInt8			reserved2:8;
	UInt8			eSATA15:1;
	UInt8			LooseSB2:1;
	UInt8			reserved3:6;
	UInt8			enableCDEject:1;
	UInt8			CDMediaVaild:1;
	UInt8			reserved4:6;
	UInt8			reserved5:8;
	UInt8			reserved6:8;
	UInt8			powerLEDBrite:8;
	UInt8			backlightBrite:8;
	UInt8			whiteOnBlack:1;
	UInt8			reserved7:7;
	UInt8			reserved8:8;
#else
	UInt8			reserved8:8;
	UInt8			reserved7:7;
	UInt8			whiteOnBlack:1;
	UInt8			backlightBrite:8;
	UInt8			powerLEDBrite:8;
	UInt8			reserved6:8;
	UInt8			reserved5:8;
	UInt8			reserved4:6;
	UInt8			CDMediaVaild:1;
	UInt8			enableCDEject:1;
	UInt8			reserved3:6;
	UInt8			LooseSB2:1;
	UInt8			eSATA15:1;
	UInt8			reserved2:8;
	UInt8			signature:8;
	UInt8			pageLength:8;
	UInt8			ParamSavable:1;
	UInt8			reserved1:1;
	UInt8			pageCode:6;
#endif
} MODE_PARAMERTER_OPERATIONS_PAGE;

typedef struct __attribute__((packed)){
	UInt8			signature;
	UInt8			reserved1[2];
    
#ifdef __LITTLE_ENDIAN__
	UInt8			OldDefault:1;
	UInt8			reserved2:3;
	UInt8			NewDefault:1;
	UInt8			reserved3:3;
#else
	UInt8			reserved3:3;
	UInt8			NewDefault:1;
	UInt8			reserved2:3;
 	UInt8			OldDefault:1;   
#endif
    
	UInt8			reserved4[2];
	UInt8			PasswordLength[2];
	UInt8			OldPassword[32];
	UInt8			NewPassword[32];
} CHANGE_ENCRYPTION_PASSPHRASE_PARAM;

typedef struct __attribute__((packed)){
	UInt8			signature;
	UInt8			reserved[5];
	UInt8			PasswordLength[2];
	UInt8			Password[32];
} CHANGE_ENCRYPTION_UNLOCK_PARAM;

typedef struct __attribute__((packed)){
	UInt8		signature;
	UInt8		reserved1[2];
	
#ifdef __LITTLE_ENDIAN__
	UInt8		combine:1;
	UInt8		reserved2:7;
#else
	UInt8		reserved2:7;
	UInt8		combine:1;
#endif
	
	UInt8		cipherID;
	UInt8		reserved3;
	UInt16		keyLength;
	UInt8		Key[32];
} RESET_DEK_PARAM;

typedef struct __attribute__((packed)){
#ifdef __LITTLE_ENDIAN__
	UInt8	pageCode:6;
	UInt8	reserved1:1;
	UInt8	ParamSavable:1;
#else
	UInt8	ParamSavable:1;
	UInt8	reserved1:1;
	UInt8	pageCode:6;
#endif
	
	UInt8	pageLength:8;
	UInt8	signature:8;
	UInt8	reserved2:8;
	
#ifdef __LITTLE_ENDIAN__
	UInt8	DisableSES:1;
	UInt8	DisableCDROM:1;
	UInt8	Reserved3:2;
    UInt8   DisableUAS:1;
    UInt8   DisableU1U2:1;
    UInt8   Reserved4:1;
	UInt8	DisableAP:1;
#else
	UInt8	DisableAP:1;
    UInt8   Reserved4:1;
    UInt8   DisableU1U2:1;
    UInt8   DisableUAS:1;
	UInt8	Reserved3:2;
	UInt8	DisableCDROM:1;
	UInt8	DisableSES:1;
#endif
	
#ifdef __LITTLE_ENDIAN__
	UInt8	DisableWhiteList:1;
	UInt8	TwoTBLimit:1;
	UInt8	reserved5:6;
#else
	UInt8	reserved5:6;
	UInt8	TwoTBLimit:1;
	UInt8	DisableWhiteList:1;
#endif
	
	UInt8	reserved6:8;
	UInt8	reserved7:8;
} DeviceConfigurationPage;

typedef struct __attribute__((packed)){
    UInt8           PageCode;
    UInt8           reserved1;
    UInt16          PageLength;
#ifdef __LITTLE_ENDIAN__
    UInt8           PowerOff:1;
    UInt8           RestartLights:1;
    UInt8           StopLights:1;
    UInt8           Obsolete:1;
    UInt8           ForceFanOn:1;
    UInt8           ReEnum:1;
    UInt8           reserved2:2;
#else
    UInt8           reserved2:2;
    UInt8           ReEnum:1;
    UInt8           ForceFanOn:1;
    UInt8           Obsolete:1;
    UInt8           StopLights:1;
    UInt8           RestartLights:1;
    UInt8           PowerOff:1;
#endif
    UInt8           unused[3];  // padding to 64 bit boundry.

} SCSI_Power_Down_page_80;

typedef struct __attribute__((packed)){
    UInt8           PageCode;
    UInt8           reserved1;
    UInt16          PageLength;
#ifdef __LITTLE_ENDIAN__
    UInt8           reserved2:2;
    UInt8           PowerButton:1;
    UInt8           reserved3:5;
#else
    UInt8           reserved3:5;
    UInt8           PowerButton:1;
    UInt8           reserved2:2;
#endif
    UInt8           unused[3];  // padding to 64 bit boundry.

} SCSI_Power_Down_Status_page_80;

typedef struct __attribute__((packed)){
	UInt8			PageCode;
	UInt8			reserved1;
	UInt8			PageLength[2];
	UInt8			DiagnosticDriveID;
	UInt8			SmartStatusMSB;
	UInt8			SmartStatusLSB;
} SCSI_Smart_Status_page_84;

typedef struct  __attribute__((aligned (8), packed)){
    UInt8			PageCode;
    UInt8			reserved1;
    UInt8			PageLength[2];
    UInt8			DiagnosticDriveID;
    UInt8			SmartStatusMSB;
    UInt8			SmartStatusLSB;
} SCSI_Smart_Status_page_84_alingned8;

typedef struct __attribute__((packed)){
	UInt8			PageCode;
	UInt8			reserved1;
	UInt8			PageLength[2];
	UInt8			DiagnosticDriveID;
	UInt8			reserved2[3];
	UInt8			SmartData[512];
} SCSI_Smart_data_page_85;

typedef struct __attribute__((packed))
{
#ifdef __LITTLE_ENDIAN__
	UInt8	ParameterCode[2];
	
	UInt8	FormatAndLinking:2;
	UInt8	TMC:2;
	UInt8	ETC:1;
	UInt8	TSD:1;
	UInt8	Obsolete:1;
	UInt8	DU:1;
	
	UInt8	ParameterLength;
	
	UInt8	SelfTestResults:4;
	UInt8	Reserved1:1;
	UInt8	SelfTestCode:3;
	
	UInt8	SelfTestNumber;
	UInt8	AccumulatedPowerOnHours[2];
	UInt8	AddressOfFirstFailure[8];
	UInt8	SenseKey:4;
	UInt8	Reserved2:4;
	UInt8	ASC;
	UInt8	ASCQ;
	UInt8	VendorSpecific;
#else
	UInt8	ParameterCode[2];
	
	UInt8	DU:1;
	UInt8	Obsolete:1;
	UInt8	TSD:1;
	UInt8	ETC:1;
	UInt8	TMC:2;
	UInt8	FormatAndLinking:2;
	
	UInt8	ParameterLength;
	
	UInt8	SelfTestCode:3;
	UInt8	Reserved1:1;
	UInt8	SelfTestResults:4;
	
	UInt8	SelfTestNumber;
	UInt8	AccumulatedPowerOnHours[2];
	UInt8	AddressOfFirstFailure[8];
	UInt8	SenseKey:4;
	UInt8	Reserved2:4;
	UInt8	ASC;
	UInt8	ASCQ;
	UInt8	VendorSpecific;
#endif
} SelfTestResultsLogParameter;

typedef struct __attribute__((packed))
{
#ifdef __LITTLE_ENDIAN__
	UInt8	PageCode:6;
	UInt8   SPF:1;
	UInt8	DS:1;
	UInt8	SubPageCode;
	UInt8	PageLength[2];
	SelfTestResultsLogParameter SelfTestLog[20];
#else
	UInt8	DS:1;
	UInt8   SPF:1;
	UInt8	PageCode:6;
	UInt8	SubPageCode;
	UInt8	PageLength[2];
	SelfTestResultsLogParameter SelfTestLog[20];
#endif
} SelfTestResultsData;

typedef struct __attribute__((packed))
{
    uint8_t     pageCode;
    uint8_t     reserved1;
    uint16_t    pageLength;
    uint8_t     temperatureCondition;   // in lower 2 bits--others are reserved
    uint8_t     reserved3;
    uint16_t    fanRPM;
    uint16_t    fanGoalPWM;
    uint16_t    currentFanPWM;
} TemperatureConditionData;

typedef struct {
	UInt8	modeParamHeader[4];
    
#ifdef __LITTLE_ENDIAN__
	UInt8	pageCode:6;
	UInt8	reserved1:1;
	UInt8	ParamSavable:1;
#else
	UInt8	ParamSavable:1;
	UInt8	reserved1:1;
	UInt8	pageCode:6;
#endif
	
	UInt8	pageLength:8;
	
	
#ifdef __LITTLE_ENDIAN__
	UInt8	standbyY:1;
	UInt8	reserved2:7;
#else
	UInt8	reserved2:7;
	UInt8	standbyY:1;
#endif
    
#ifdef __LITTLE_ENDIAN__
	UInt8	standbyZ:1;
	UInt8	idleA:1;
	UInt8	idleB:1;
	UInt8	idleC:1;
	UInt8	reserved3:4;
#else
	UInt8	reserved3:4;
	UInt8	idleC:1;
	UInt8	idleB:1;
	UInt8	idleA:1;
	UInt8	standbyZ:1;
#endif
	
	UInt8	idleAContionTimer[4];
	UInt8	standbyZConditionTimer[4];
	UInt8	idleBContionTimer[4];
	UInt8	idleCContionTimer[4];
	UInt8	standbyYConditionTimer[4];
    uint8   reserved4[16];
} PowerConditionPage;

typedef struct __attribute__((packed)){
    UInt64      ReturnedLogicalBlockAddress;
    UInt32      LogicalBlockLength;
    UInt8       ProtEN:1;
    UInt8       PType:3;
    UInt8       PIExponet:4;
    UInt16      LowestAlibgnedLBA:14;
    UInt16      LBPRZ:1;
    UInt16      LBPME:1;
    UInt8       reserved[16];
} ReadCapacityPage;

enum {
	kMaxBSDNameLength			=	MAXPATHLEN,										// Max length of the bsd name as a c string 
};

//enum {
//    kSEND_DIAGNOSTIC_COMMAD             = 0x1D,
//	kDIAG_SHORT_TEST_COMMAND			= 0x1D,
//	kDIAG_EXTENDED_TEST_COMMAND			= 0x1D,
//	kDIAG_GET_TEST_STATUS_COMMAND		= 0x1C,
//};
//
//enum {
//    kSEND_DIAGNOSTIC_COMMAND_LENGTH     = 6,
//	kDIAG_SHORT_TEST_COMMAND_LENGTH		= 6,
//	kDIAG_EXTENDED_TEST_COMMAND_LENGTH	= 6,
//	kDIAG_GET_TEST_STATUS_COMMAND_LENGTH = 6
//};

// Moved to WDDMcommon.h
//
//typedef struct {
//	UInt8		LastHandyBlockAddress[4];
//	UInt8		BlockLength[4];
//	UInt8		Reserved1[2];
//	UInt8		MaximumTransferLength[2];
//} ReadHandyCapacityData;

//enum {
//	kHandStoreSecurityBlock		=	1,
//	kHandStoreUserSettingBlock	=	2
//};
//
//typedef union{
//	UInt8	byte[512];
//	struct {
//		UInt8	Signature[4];
//		UInt8	reserved1[4];
//		UInt8	IterationCount[4];
//		UInt8	Salt[8];
//		UInt8	reserved2[4];
//		UInt8	hint[202];
//      UCHAR   PasswordGUID[82];  // (maximum 40 characters plus NULL ending)
//		UInt8	reserved3[203];
//		UInt8	checksum;
//	}data; 
//} SecurityBlock;

typedef union __attribute__((packed)){
	UInt8	byte[512];
	struct __attribute__((packed)){
		UInt8	Signature[4];	//	  4 Bytes
		UInt8	reserved1[4];	//	  4 Bytes
		UInt8	Label[64];		//	 64 Bytes
		UInt8	reserved[439];	//	439 Bytes
		UInt8	checksum;		//	  1 Byte
	}data; 
} HandyStoreUserBlock;


typedef struct __attribute__((packed)){
	UInt8	modeParamHeader[4];
	
#ifdef __LITTLE_ENDIAN__
	UInt8	pageCode:6;
	UInt8	reserved1:1;
	UInt8	ParamSavable:1;
#else
	UInt8	ParamSavable:1;
	UInt8	reserved1:1;
	UInt8	pageCode:6;
#endif
	
	UInt8	pageLength:8;
	UInt8	signature:8;
	UInt8	reserved2:8;
	
#ifdef __LITTLE_ENDIAN__
	UInt8	eSATA15:1;
	UInt8	LooseSB2:1;
	UInt8	reserved3:6;
#else
	UInt8	reserved3:6;
	UInt8	LooseSB2:1;
	UInt8	eSATA15:1;
#endif
	
#ifdef __LITTLE_ENDIAN__
	UInt8	enableCDEject:1;
	UInt8	CDMediaVaild:1;
	UInt8	reserved4:6;
#else
	UInt8	reserved4:6;
	UInt8	CDMediaVaild:1;
	UInt8	enableCDEject:1;
#endif
	
	UInt8	reserved5:8;
	UInt8	reserved6:8;
	UInt8	powerLEDBrite:8;
	UInt8	backlightBrite:8;
	
#ifdef __LITTLE_ENDIAN__
	UInt8	whiteOnBlack:1;
	UInt8	reserved7:7;
#else
	UInt8	reserved7:7;
	UInt8	whiteOnBlack:1;
#endif
	
	UInt8	reserved8:8;
} OperationsPage;

typedef struct __attribute__((packed)){
    UInt64  logicalBlbokAddress;
    UInt32  numberofLogicalBlocks;
    UInt32  reserved;
} UNMAP_BLOCK_DESCRIPTOR;

typedef struct __attribute__((packed)){
    UInt16  unmapDataLengh;
    UInt16  unmapBlockDescriptorDataLength;
    UInt8   Reserved[4];
    UInt8   unmapBlockDescriptor[];
} UNMAP_PARAMETER_LIST;

@interface WDDevice (SCSICommands)

- (BOOL) SCSITestUnitReady:(NSMutableDictionary **)errorDict;
- (BOOL) SCSIStartUnit:(NSMutableDictionary **)errorDict;
- (int) GetSCSIInquiryData: (SCSICmd_INQUIRY_WDDataPtr) buffer buffersize:(int)bufferSize error:(NSMutableDictionary **)errorDict;
- (int) GetSCSISerialNumber: (UInt8 *) buffer buffersize:(int)bufferSize error:(NSMutableDictionary **)errorDict;
- (int) GetSCSIFirmwareVersion: (UInt8 *) buffer buffersize:(int)bufferSize error:(NSMutableDictionary **)errorDict;

//- (int) EncryptArmWithPassword:(UInt8*)  cookedPassword  passwordLength:(UInt8) passwordLength error:(NSMutableDictionary **)errorDict;
//- (int) EncryptDisArmWithPassword:(UInt8*) cookedPassword  passwordLength:(UInt8) passwordLength error:(NSMutableDictionary **)errorDict;
//- (int) EncryptChangePassword: (UInt8)passwordLength oldCookedPassword:(UInt8*) oldCookedPassword newCookedPassword:(UInt8*) newCookedPassword error:(NSMutableDictionary **)errorDict;
//- (int) EncryptGetStatus:(EncryptStatusReturnData *) returnData error:(NSMutableDictionary **)errorDict;
//- (int) EncryptUnlockWithPassword:(UInt8*)password passwordLength:(UInt8) passwordLength  error:(NSMutableDictionary **)errorDict;
//- (int) EncryptResetDEK:(UInt32 *) newKey error:(NSMutableDictionary **)errorDict;
//- (int) ReadHandyStoreBlockNumber:(int) blockNumber blockdata:(UInt8 *) blockData error:(NSMutableDictionary **)errorDict;
//- (int) WriteHandyStoreBlockNumber:(int) blockNumber blockdata:(UInt8 *) blockData error:(NSMutableDictionary **)errorDict;
//- (int) GetHandyStoreCapacity: (ReadHandyCapacityData*) HandyCapacityData error:(NSMutableDictionary **)errorDict;

//
//  Destination Disk LUN compatable commands
//
- (int) EncryptArmWithPassword:(UInt8*)  cookedPassword  passwordLength:(UInt8) passwordLength onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) EncryptDisArmWithPassword:(UInt8*) cookedPassword  passwordLength:(UInt8) passwordLength onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) EncryptChangePassword: (UInt8)passwordLength oldCookedPassword:(UInt8*) oldCookedPassword newCookedPassword:(UInt8*) newCookedPassword onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) EncryptGetStatus:(EncryptStatusReturnData *) ReturnData onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) EncryptUnlockWithPassword:(UInt8*)password passwordLength:(UInt8) passwordLength  onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) EncryptResetDEK:(UInt32 *) newKey onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) ReadHandyStoreBlockNumber:(int) blockNumber blockdata:(UInt8 *) blockData onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) WriteHandyStoreBlockNumber:(int) blockNumber blockdata:(UInt8 *) blockData onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
- (int) GetHandyStoreCapacity: (ReadHandyCapacityData*) HandyCapacityData onLUN: (UInt32)destLUN error:(NSMutableDictionary **)errorDict;
//
//
- (int) UnmapSMRDrive: (NSMutableDictionary **)errorDict;

//

- (int) verifyHandyStoreSecurityBlock:(SecurityBlock *)blockData;
- (int) initHandyStoreSecurityBlock:(SecurityBlock*) blockData;
- (int) verifyHandyStoreUserBlock:(HandyStoreUserBlock *)blockData;
- (int) initHandyStoreUserBlock:(HandyStoreUserBlock*) blockData;

- (int) GetRawDeviceCapacity:(UInt64 *)capacity error:(NSMutableDictionary **)errorDict;

- (int) setVCDState: (bool) state error:(NSMutableDictionary **)errorDict;
- (BOOL) vcdStatus:(NSMutableDictionary **)errorDict;
- (bool) VCDSupported:(NSMutableDictionary **)errorDict;
- (int)  performVCDConfiguration;
- (int) ReadVirturalCDCapacity:(READ_VITURAL_CD_CAPACITY_DATA *) VCDCapacityDataPtr;

- (int) WriteVirtualCD:(unsigned char *) DataPtr onSlot: (UInt64)destSlot atBlock:(UInt64)blockAddress length:(UInt64)dataLength;
- (int) ReadVirtualCD:(unsigned char *) DataPtr onSlot: (UInt64)destSlot atBlock:(UInt64)blockAddress length:(UInt64)dataLength;
- (int) ModeSelectCommand:(unsigned char *) PageDataPtr length:(UInt64)PageDataLength savepages:(bool)savePages;
- (int) ModeSenseCommand:(unsigned char *) PageDataPtr length:(UInt64)PageDataLength pagecontrol:(int)pageControl pagecode:(int)pageCode;

- (int) diagStartShortTest:(NSMutableDictionary **)errorDict;
- (int) diagStartExtTest:(NSMutableDictionary **)errorDict;
- (int) diagAbortTest:(NSMutableDictionary **)errorDict;
- (int) diagGetSmartData:(ATASMARTData *) smartData forSlot: (int) slot error:(NSMutableDictionary **)errorDict;
- (int) diagGetSmartStatus:(UInt16 *) smartStatus forSlot: (int)slot error:(NSMutableDictionary **)errorDict;
- (int) getSelfTestData:(SelfTestResultsData *) selfTestData forSlot: (int)slot error:(NSMutableDictionary **)errorDict;
- (int) diagReEnumerate:(NSMutableDictionary **)errorDict;

- (int) ConfigSetDriveSleepTimer:(UInt8) minutes error:(NSMutableDictionary **)errorDict;
- (int) ConfigGetDriveSleepTimer:(UInt8 *) minutes error:(NSMutableDictionary **)errorDict;

- (int)reportRAIDStatus:(UInt8 *) buffer buffersize:(UInt32)bufferSize sourceID: (UInt8) sourceID error:(NSMutableDictionary **)errorDict;
- (int)reportSlotDetails:(UInt8 *) buffer buffersize:(UInt32)bufferSize sourceID: (UInt8) sourceID error:(NSMutableDictionary **)errorDict;
- (int)setHWRAIDConfiguration:(UInt8 *) buffer buffersize:(UInt32)bufferSize error:(NSMutableDictionary **)errorDict;
- (int)reportLUNs:(UInt8 *) buffer buffersize:(UInt32)bufferSize error:(NSMutableDictionary **)errorDict;
- (int) getTemperatureCondition: (TemperatureConditionData*) tempCondition error: (NSMutableDictionary**) errorDict;

- (BOOL) getLEDStatus:(NSMutableDictionary **)errorDict;
- (int)  setPowerLED: (BOOL) LEDOn;

- (int) getPowerOffStatus: (UInt16*)pageLength error:(NSMutableDictionary **)errorDict;
- (int) powerOff:(NSMutableDictionary **)errorDict;

@end
